 ////////////////////////////////////////////////////////////////////////////////////////////
//   __  __  __  //  JoyPlug 1 v1.00f0     Released on March 25, 1999                      //
//   \ \ \ \ \/  //  a joystick controlled, interactive, real-time lowpass resonant filter //
//    \_\ \_\    //  for Buzz 1.  Uses my own homebrew DSP code and the standard API that  //
// Ken           //  ships with Buzz for audio processing and uses the Windows Joystick API//
// "Wirehead"    //  (not DirectInput -- It does a bad job in this kind of situation) for  //
// Wronkiewicz   //  realtime I/O.  Compiles under VC++ 6.0, with no warnings, even!       //
/////////////////////////////////////////////////////////////////////////////////////////////
// License:  Some of this code is provided by the creators of Buzz.  That is covered under //
// the standard Buzz licensing.  All of the DSP code that I created is insanely simple, so //
// I can't really restrict the use of it without laying claim to every implimentation of a //
// lowpass resonant filter.  Therefore, the license is this:  Feel free to use my code in  //
// your own projects.  A greet or thanks would be nice.  None of the other plugin authors  //
// have been nice enough to give out their source.  Were they to have done so, my coding   //
// would have gone much smoother.  But they didn't, so I will.  Don't make me regret doing //
// this -- if I see somebody making a direct rip of my code and putting their name on it,  //
// I will not repeat this generosity to the Buzz community.                                //
/////////////////////////////////////////////////////////////////////////////////////////////
// http://www.wirewd.com/wh/       mailto:wh@wirewd.com    Will hack audio for ca$h!       //
////////////////////////////////////////////////////////////////////////////////////////////

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <math.h>
#include <windows.h>
#include <mmsystem.h>


// You may have to change the location of this, depending on where your buzz directory is.
// You also may have to adjust the paramaters for where a library is in the build settings.
#include "MachineInterface.h"

#define MAX_TRACKS      16

#pragma pack()

//This is the machine info structure.  This lets Buzz know what's up with the specific generator.
CMachineInfo const MacInfo = 
{
        MT_EFFECT,							// type, MT_GENERATOR or MT_EFFECT
        MI_VERSION,							// version, always set this to MI_VERSION
		0,									// Flags
        0,									// minimum number of tracks
        0,									// maximum number of tracks
        0,									// number of global parameters
        0,									// number of track parameters
        NULL,								// pointer to list of parameters
        0,									// number of attributes
        NULL,								// pointer to list of attributes
#ifdef _DEBUG
        "Joystick Filter 1 (debug build)",  // name for debug build
#else
        "Joystick Filter 1",                // name for release build
#endif
        "JoyFilt1",							// short name
        "Ken \"Wirehead\" Wronkiewicz",		// your name
        NULL								// list of commands 
};

struct WFState  //Saves the state of the filter.  I'm using doubles where floats might be a
				//little faster, but I found that the performance decrease was negligable for
				//higher quality
{
		double c;							//Filter coefficent -- precalculated
		double pos;							//Filter position -- calculated for each var
		double speed;						//Filter speed -- calculated for each var
		double r;							//Filter resonance -- between 0 and 1
		double dist;						//Filter distortion coefficent -- a multiplier
		int samplerate;						//Sample rate -- usually 44100, but it's not nice to
											//make stupid assumptions
};

//This is the actual worker class for the program.  All of the important stuff is handled
//through here.
class mi : public CMachineInterface
{
public: 
        mi();												//Constructor
        virtual ~mi();										//Destructor -- must be virtual 

        virtual void Init(CMachineDataInput * const pi);	//Buzz API: Init the effect
        virtual void Tick();								//Buzz API: Called each tick
        virtual bool Work(float *psamples, int numsamples, int const mode);
															//Buzz API: Actually process a signal
        virtual void SetNumTracks(int const n);				//Buzz API: Set the number of tracks

private:        
		void UpdateStick();									//Update the joystick position
		void Calc_Coeff(WFState *theState, double g, double q, double d);
															//Calculate coefficents
		void Init_Filt(WFState *theState);					//Init the filter
		void Filt_Work(WFState *theState, float *psamples, int numsamples);
															//Make the filter do work
		JOYCAPS		theStickCaps;							//Capabilities of the joystick
		JOYINFO		theStickInfo;							//Info for the joystick
		UINT		stickPos1;								//Axis one for the joystick
		UINT		stickPos2;								//Axis two for the joystick
		UINT		stickPos3;								//Axis three for the joystick
#ifdef _DEBUG
		bool		showInfo;								//Debug variable for showing the info
		bool		isWorking;								//Debug variable for showing status
#endif
		WFState		dspState;								//Current DSP status
};

DLL_EXPORTS

//These are the internal procedures for my DSP code.

void mi::Init_Filt(WFState *theState) //Initialize the filter.  theState is the state struct
{
	theState->pos = 0; //Zero the position
	theState->speed = 0; //Zero the speed
	theState->samplerate = pMasterInfo->SamplesPerSec; //Set the samplerate
}

void mi::Calc_Coeff(WFState *theState, double freq, double q, double d)
	//Calculate the coefficents.  theState is the current state struct.  freq is the cutoff
	//frequency in Hz  q is the resonance in fractions of 1  d is the distortion, a constant
	//multiplier.
{
	theState->c = (2-2*cos(2*PI*freq / theState->samplerate));	//Calculate the coefficent
	theState->r = q; //Store resonence
	theState->pos = 0; //Clear position
	theState->speed = 0; //Clear speed
	theState->dist = 10*d+1; //Set distortion.
}

void mi::Filt_Work(WFState *theState, float *psamples, int numsamples)
	//Actually do some work with the filter.  theState is the current state struct.  *psamples
	//is an array of floating point samples, numsamples is the number of samples to process.
	//Just for the heck of it, I decided to use double floating point variables, which gives
	//us insane amounts of accuracy (Beyond 32-bit, not including the exponents) for not much
	//of a speed hit and good quality.  I use a PII/300, which means it may suck for those who
	//are not using as fast of a machine.
{
	int		count; //Where we are in the array
		
	for(count = 0; count < numsamples ; count++) {
		theState->speed = theState->speed + (psamples[count] - theState->pos) * theState->c;
        theState->pos = theState->pos + theState->speed;
        theState->speed = theState->speed * theState->r;
        psamples[count] = (float)(theState->dist*theState->pos);
#ifdef _DEBUG
		isWorking = true;
#endif
	}
}

void mi::UpdateStick()
{
		//Reads in the position of the joystick and updates the filter coefficents
	    //based on the results.  This is my code.
		MMRESULT	theResult; //The result code
#ifdef _DEBUG
		char		pozText[200]; //Some text for debugging purposes
		int			len;		//The length of the debugging text
#endif
		double		floatPos2; //floating point version of the joystick paramater for processing

		theResult = joyGetPos(JOYSTICKID1,&theStickInfo);
		if (theResult == JOYERR_NOERROR) {
			//Read the stick position data
			stickPos1 = theStickInfo.wXpos;
			stickPos2 = theStickInfo.wYpos;
			stickPos3 = theStickInfo.wZpos;
			//Process the data for the second axis (controls resonance)
			floatPos2 = (double)stickPos2; //First cast the integer to a double
			floatPos2 = 32767.0 - abs((int)stickPos2 - 32767); //This makes the center the 
				//position of highest resonance and makes the left and right mirrors of each other
			floatPos2 = floatPos2/32767; //Divide it up so that the resonance paramater is 
				// between 1 and 0
			floatPos2 = floatPos2*.25 + .75; //Bias the resonance setting to sit in the middle
#ifdef _DEBUG
			//Print out info, if we need it
			if ((theStickInfo.wButtons) & JOY_BUTTON4 !=  0) {
				if (!showInfo) {
					if (isWorking) {
						len = sprintf(pozText, "Processing / %d %d %d",stickPos1,stickPos2, stickPos3);
					} else {
						len = sprintf(pozText, "Not Processing / %d %d %d",stickPos1,stickPos2, stickPos3);
					}
					pCB->MessageBox(pozText);
					showInfo = true;
				}
			} else {
				showInfo = false;
			}
#endif
			//Calculate the resonance coefficents for the filter and save
			Calc_Coeff(&dspState, (double)stickPos1*0.10, floatPos2, (double)stickPos3/65535.0);
		} else {
			//Error messages
			switch( theResult )
			{
			case MMSYSERR_NODRIVER: pCB->MessageBox("Joystick Error: No Driver");
									break;
			case MMSYSERR_INVALPARAM: pCB->MessageBox("Joystick Error: Invalad Paramater");
									  break;
			case JOYERR_UNPLUGGED: pCB->MessageBox("Joystick Error: Joystick Unplugged");
								   break;
			}
		}
	}



// These are the exported procedures for the Buzz API.

mi::mi()        
// intialize pointers to your values here, and zero stuff.
{
	//Clear all paramaters, because we don't use any.
	GlobalVals = NULL;
	TrackVals = NULL;
	AttrVals = NULL;
	//Zero the stick
	stickPos1 = 0;
	stickPos2 = 0;
#ifdef _DEBUG
	showInfo = FALSE;
#endif
}

mi::~mi()
	// free any memory you allocated here.  We allocated nothing, so we do nothing.
{
	// Nothing!
}


void mi::Init(CMachineDataInput * const pi)
{
    // initialize the machine
    // you can ignore pi if you haven't implemented the 'Save' method.  We don't
	// have any variables to save, but it's used to retrieve saved paramaters.
	UINT		numDevs; // Number of joystick
	MMRESULT	theResult;//Joystick library error value

	numDevs = joyGetNumDevs();
	if(numDevs = 0) {
			pCB->MessageBox("No Joystick Attached");
	} else {
			theResult = joyGetDevCaps(JOYSTICKID1,&theStickCaps,sizeof(theStickCaps));
			if (theResult != JOYERR_NOERROR) {
				pCB->MessageBox("Error on joystick");
			} else {
#ifdef _DEBUG
				pCB->MessageBox("Joystick Available");
#endif
			}
	}
	Init_Filt(&dspState);
	UpdateStick();
}

// this is called by buzz after Init and every time the user adds/deletes a track.
// Since we have no tracks, there's not much point in doing anything here.
void mi::SetNumTracks(int const n)
{
	//Nothing
}

// This is called by buzz after each tick.  JoyPlug1 has no need for any paramaters,
// so there's no point in doing anything here.
void mi::Tick()
{
	//Nothing!
}

bool mi::Work(float *psamples, int numsamples, int const mode)
{        
    // Generate 'numsamples' samples to 'psamples' here.  For those who don't know the 
	// buzz API and are too lazy to discover it for themselves, the psamples array is an
	// array of floating point values.  They range from -65535 to +65535.  This gives you 
	// roughly 24 bit accuracy, with some extra accuracy on the bottom end for quiet signals,
	// which is pretty respectible.
	// The mode constant is not IOTTMCO.  I'm not sure if I got it right, so I'll warn you
	// now about that.
	// This is the main part of the Buzz API that actually does stuff.

	if (mode == WM_READWRITE) {
		UpdateStick();
#ifdef _DEBUG
		isWorking = false;
#endif
		Filt_Work(&dspState, psamples, numsamples);
		return (true);
	} else {
		return (false);
	}
}
