
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <math.h>
#include <float.h>
#include "../MachineInterface.h"
#include "../dsplib/dsplib.h"

#pragma optimize ("a", on)

#define MAX_TRACKS				8

#define EGS_ATTACK				0
#define EGS_SUSTAIN				1 
#define EGS_RELEASE				2
#define EGS_NONE				3

#define MIN_AMP					(0.0001 * (32768.0 / 0x7fffffff))

double const oolog2 = 1.0 / log(2);

CMachineParameter const paraWForm = 
{ 
	pt_byte,										// type
	"Waveform",
	"Shape of waveform",							// description
	1,												// MinValue	
	7,												// MaxValue
	0,												// NoValue
	MPF_STATE,										// Flags
	1
};

CMachineParameter const paraAttack = 
{ 
	pt_word,										// type
	"Attack",
	"Attack half life in ms",						// description
	1,												// MinValue	
	0x0800,											// MaxValue
	0,												// NoValue
	MPF_STATE,										// Flags
	500
};

CMachineParameter const paraDecay = 
{ 
	pt_word,										// type
	"Decay",
	"Decay half life in ms",						// description
	1,												// MinValue	
	0x0800,											// MaxValue
	0,												// NoValue
	MPF_STATE,										// Flags
	500
};

CMachineParameter const paraEffectLow = 
{ 
	pt_word,										// type
	"low effect",
	"Fractal Effect (Low)",							// description
	0,												// MinValue	
	0xfffe,											// MaxValue
	0xffff,											// NoValue
	MPF_STATE,										// Flags
	0x8000,
};

CMachineParameter const paraEffectHigh = 
{ 
	pt_word,										// type
	"high effect",
	"Fractal Effect (High)",							// description
	0,												// MinValue	
	0xfffe,											// MaxValue
	0xffff,											// NoValue
	MPF_STATE,										// Flags
	0x8000,
};

CMachineParameter const paraDepth = 
{ 
	pt_byte,										// type
	"depth",
	"Fractal Depth",								// description
	0,												// MinValue	
	10,												// MaxValue
	0xff,											// NoValue
	MPF_STATE,										// Flags
	0x01
};

CMachineParameter const paraNote = 
{ 
	pt_note,										// type
	"Note",
	"Note",											// description
	NOTE_MIN,										// MinValue	
	NOTE_OFF,										// MaxValue
	NOTE_NO,										// NoValue
	0,												// Flags
	0
};

CMachineParameter const paraVolume = 
{ 
	pt_byte,										// type
	"Volume",
	"Volume [sustain level] (0=0%, 80=100%, FE=~200%)",	// description
	0,												// MinValue	
	0xfe,  											// MaxValue
	0xff,    										// NoValue
	MPF_STATE,										// Flags
	0x80
};


CMachineParameter const *pParameters[] = { 
	// global
	&paraWForm,
	&paraAttack,
	&paraDecay,
	&paraEffectLow,
	&paraEffectHigh,
	&paraDepth,
	// track
	&paraNote,
	&paraVolume
};

#pragma pack(1)

class gvals
{
public:
	byte wform;
	word attack;
	word decay;
	word effect_low;
	word effect_high;
	byte depth;

};

class tvals
{
public:
	byte note;
	byte volume;

};

#pragma pack()

CMachineInfo const MacInfo = 
{
	MT_GENERATOR,							// type
	MI_VERSION,
	0,										// flags
	1,										// min tracks
	MAX_TRACKS,								// max tracks
	6,										// numGlobalParameters
	2,										// numTrackParameters
	pParameters,
	0, 
	NULL,
#ifdef _DEBUG
	"Soft Tone 2 (Debug build)",			// name
#else
	"Soft Tone 2",
#endif
	"Softy2_",								// short name
	"Steve Horne", 						// author
	NULL
};

class mi;

class mi;

class CTrack
{
public:
	void Tick(tvals const &tv);
	void Stop();
	void Reset();
	void Generate(float *psamples, int numsamples);

    double Fractal (double p, double q);

	void Sine     (float *psamples, int numsamples);
	void Square   (float *psamples, int numsamples);
	void Triangle (float *psamples, int numsamples);
	void Ramp     (float *psamples, int numsamples);
	void Hex      (float *psamples, int numsamples);
	void Jaggy    (float *psamples, int numsamples);
	void AltRamp  (float *psamples, int numsamples);

public:

	double Freq;
	double TargetAmp;
	int    NoteOn;

	double CurrPhase;
	double CurrAmp;

	mi *pmi;
};

class mi : public CMachineInterface
{
public:
	mi();
	virtual ~mi();

	virtual void Init(CMachineDataInput * const pi);
	virtual void Tick();
	virtual bool Work(float *psamples, int numsamples, int const mode);
	virtual void SetNumTracks(int const n) { numTracks = n; }
	virtual void Stop();

public:
	int numTracks;
	CTrack Tracks[MAX_TRACKS];

	tvals tval[MAX_TRACKS];
	gvals gval;

	byte   WForm;
	double DecayA;
	double AttackA;
	double AttackB;

	double EffectHigh;
	double EffectLow;

	byte Depth;
};

DLL_EXPORTS

mi::mi()
{
	GlobalVals = &gval;
	TrackVals  = tval;
	AttrVals   = NULL;
}

mi::~mi()
{

}

void CTrack::Reset()
{
	NoteOn    = 0;
	Freq      = 0.0;
	TargetAmp = 1.0;
	CurrPhase = 0.0;
	CurrAmp   = 0.0;
}

double CTrack::Fractal (double p, double q)
{
	float s = (p + 1.0) / 2.0;

	double Effect  = pmi->EffectLow + ((pmi->EffectHigh - pmi->EffectLow) * q);
	double EffectB = 3.0 - (3.0 * Effect);
	double EffectA = 1.0 - (Effect + EffectB);

	if (s < 0.0)
	{
		s = 0.0;
	}
	else if (s > 1.0)
	{
		s = 1.0;
	}

	float ss;
	int n = pmi->Depth;

	while (n-- > 0)
	{
		ss = s * s;

		s =   (EffectA * ss * s)
			+ (EffectB * ss    )
			+ (Effect  * s     );
	}

	return (s * 2.0) - 1.0;
}


void mi::Init(CMachineDataInput * const pi)
{
	WForm = 1;

	AttackA = pow(2.0, -1000.0 / (((double) pMasterInfo->SamplesPerSec) * 500.0));
	AttackB = 1.0 - AttackA;

	DecayA = pow(2.0, -1000.0 / (((double) pMasterInfo->SamplesPerSec) * 500.0));

	EffectHigh = (((float) 32768) * 9.0) / ((float) 65534);
	EffectLow  = (((float) 32768) * 9.0) / ((float) 65534);

	Depth  = 1;

	for (int c = 0; c < MAX_TRACKS; c++)
	{
		Tracks[c].pmi = this;
		Tracks[c].Reset();
	}

}

void CTrack::Tick(tvals const &tv)
{
	if (tv.volume != paraVolume.NoValue)
		TargetAmp = (float)(tv.volume * (1.0 / 0x80));

	if (tv.note == NOTE_OFF)
	{
		NoteOn = 0;
	}
	else if (tv.note != NOTE_NO)
	{
		NoteOn = 1;

		int l_Note = ((tv.note >> 4) * 12) + (tv.note & 0x0f) - 70;

		Freq = (440.0 * pow (2.0, ((float) l_Note) / 12.0)) / pmi->pMasterInfo->SamplesPerSec;
	}
}

void CTrack::Generate(float *psamples, int numsamples)
{
	int	   c = numsamples;
	float *p = psamples;

	if (NoteOn)
	{
		do
		{
			*p = CurrAmp;
			CurrAmp = (CurrAmp * pmi->AttackA) + (TargetAmp * pmi->AttackB);
			p++;
		} while (--c > 0);
	}
	else
	{
		do
		{
			*p = CurrAmp;
			CurrAmp *= pmi->DecayA;
			p++;
		} while (--c > 0);
	}

	switch (pmi->WForm)
	{
		case 1 :
			Sine (psamples, numsamples);
			break;
		case 2 :
			Triangle (psamples, numsamples);
			break;
		case 3 :
			Jaggy (psamples, numsamples);
			break;
		case 4 :
			Hex (psamples, numsamples);
			break;
		case 5 :
			Square (psamples, numsamples);
			break;
		case 6 :
			Ramp (psamples, numsamples);
			break;
		case 7 :
			AltRamp (psamples, numsamples);
			break;
	}

	c = numsamples;
	p = psamples;

	do
	{
		*p *= 32768.0;  p++;
	} while (--c > 0);
}

void CTrack::Sine(float *psamples, int numsamples)
{
	do
	{
		*psamples *= Fractal (sin (CurrPhase * 2.0 * PI), *psamples);
		CurrPhase += Freq;
		psamples++;
	} while (--numsamples > 0);

	while (CurrPhase >= 1.0)
		CurrPhase -= 1.0;
}

void CTrack::Square(float *psamples, int numsamples)
{
	do
	{
		while ((numsamples > 0) && (CurrPhase < 0.5))
		{
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while ((numsamples > 0) && (CurrPhase < 1.0))
		{
			*psamples = -(*psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while (CurrPhase >= 1.0)
		{
			CurrPhase -= 1.0;
		}

	} while (numsamples > 0);
}

void CTrack::Triangle(float *psamples, int numsamples)
{
	do
	{
		while ((numsamples > 0) && (CurrPhase < 0.25))
		{
			*psamples *= Fractal (CurrPhase * 4, *psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while ((numsamples > 0) && (CurrPhase < 0.75))
		{
			*psamples *= Fractal (2.0 - (CurrPhase * 4), *psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while ((numsamples > 0) && (CurrPhase < 1.25))
		{
			*psamples *= Fractal ((CurrPhase * 4) - 4.0, *psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while (CurrPhase >= 1.0)
		{
			CurrPhase -= 1.0;
		}

	} while (numsamples > 0);
}

void CTrack::Ramp(float *psamples, int numsamples)
{
	do
	{
		while ((numsamples > 0) && (CurrPhase < 1.00))
		{
			*psamples *= Fractal ((CurrPhase * 2) - 1.0, *psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while (CurrPhase >= 1.0)
		{
			CurrPhase -= 1.0;
		}

	} while (numsamples > 0);
}

void CTrack::AltRamp(float *psamples, int numsamples)
{
	do
	{
		while ((numsamples > 0) && (CurrPhase < 0.5))
		{
			*psamples *= Fractal (CurrPhase * 2, *psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while ((numsamples > 0) && (CurrPhase < 1.0))
		{
			*psamples *= Fractal (1.0 - (CurrPhase * 2), *psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while (CurrPhase >= 1.0)
		{
			CurrPhase -= 1.0;
		}

	} while (numsamples > 0);
}

void CTrack::Hex(float *psamples, int numsamples)
{
	do
	{
		while ((numsamples > 0) && (CurrPhase < 0.125))
		{
			*psamples *= Fractal (CurrPhase * 8, *psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while ((numsamples > 0) && (CurrPhase < 0.375))
		{
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while ((numsamples > 0) && (CurrPhase < 0.625))
		{
			*psamples *= Fractal (4.0 - (CurrPhase * 8), *psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while ((numsamples > 0) && (CurrPhase < 0.875))
		{
			*psamples = -(*psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while ((numsamples > 0) && (CurrPhase < 1.125))
		{
			*psamples *= Fractal ((CurrPhase * 8) - 8.0, *psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while (CurrPhase >= 1.0)
		{
			CurrPhase -= 1.0;
		}

	} while (numsamples > 0);
}

void CTrack::Jaggy(float *psamples, int numsamples)
{
	do
	{
		while ((numsamples > 0) && (CurrPhase < 0.125))
		{
			*psamples *= Fractal (CurrPhase * 8, *psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while ((numsamples > 0) && (CurrPhase < 0.25))
		{
			*psamples *= Fractal (2.0 - (CurrPhase * 8), *psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while ((numsamples > 0) && (CurrPhase < 0.375))
		{
			*psamples *= Fractal ((CurrPhase * 8) - 2.0, *psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while ((numsamples > 0) && (CurrPhase < 0.625))
		{
			*psamples *= Fractal (4.0 - (CurrPhase * 8), *psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while ((numsamples > 0) && (CurrPhase < 0.75))
		{
			*psamples *= Fractal ((CurrPhase * 8) - 6.0, *psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while ((numsamples > 0) && (CurrPhase < 0.875))
		{
			*psamples *= Fractal (6.0 - (CurrPhase * 8), *psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while ((numsamples > 0) && (CurrPhase < 1.125))
		{
			*psamples *= Fractal ((CurrPhase * 8) - 8.0, *psamples);
			CurrPhase += Freq;
			psamples++;
			numsamples--;
		}

		while (CurrPhase >= 1.0)
		{
			CurrPhase -= 1.0;
		}

	} while (numsamples > 0);
}

void mi::Tick()
{
	if (gval.wform != paraWForm.NoValue)
	{
		WForm = gval.wform;
	}

	if (gval.attack != paraAttack.NoValue)
	{
		AttackA = pow(2.0, -1000.0 / (((double) pMasterInfo->SamplesPerSec) * ((double) gval.attack)));
		AttackB = 1.0 - AttackA;
	}

	if (gval.decay != paraDecay.NoValue)
	{
		DecayA = pow(2.0, -1000.0 / (((double) pMasterInfo->SamplesPerSec) * ((double) gval.decay)));
	}

	if (gval.effect_high != paraEffectHigh.NoValue)
	{
		EffectHigh = (((float) gval.effect_high) * 9.0) / ((float) 65534);
	}

	if (gval.effect_low != paraEffectLow.NoValue)
	{
		EffectLow = (((float) gval.effect_low) * 9.0) / ((float) 65534);
	}

	if (gval.depth != paraDepth.NoValue)
		Depth = gval.depth;

	for (int c = 0; c < numTracks; c++)
		Tracks[c].Tick(tval[c]);

}

bool mi::Work(float *psamples, int numsamples, int const)
{
	bool gotsomething = false;

	if (numTracks > 0)
	{
		Tracks[0].Generate(psamples, numsamples);
	}

	for (int c = 1; c < numTracks; c++)
	{
		float *paux = pCB->GetAuxBuffer();
		Tracks[c].Generate(paux, numsamples);

		DSP_Add(psamples, paux, numsamples);
	}

	return true;
}

void CTrack::Stop()
{
	NoteOn  = 0;
	CurrAmp = 0.0;
}

void mi::Stop()
{
	for (int c = 0; c < numTracks; c++)
		Tracks[c].Stop();
}
