//
//  Envelope related stuff
//

#ifndef __NR_LIB_HEAD
#define __NR_LIB_HEAD

#include <math.h>
#include <float.h>
#include "../MachineInterface.h"

#include "fractal.h"
#include "wave.h"

#define MIN_AMP	((float) 0.0005)

///////////////////////////////////////////////////////////////////////////////
//
//  Macro for buzz waveform parameter specifications
//
//  NR_LIB_ATTACK_PAR is used for attack and decay - anything that
//  has a standard e^-kt 'half-life' pattern.
//

#define NR_LIB_ATTACK_PAR(p_Var,p_Name,p_Desc, p_Def)	\
CMachineParameter const p_Var =							\
{														\
	pt_word,											\
	p_Name,												\
	p_Desc,												\
	1,													\
	0xffff,												\
	0,													\
	MPF_STATE,											\
	p_Def												\
};

#define NR_LIB_AMPLITUDE_PAR(p_Var,p_Name,p_Desc,p_Def)	\
CMachineParameter const p_Var =							\
{														\
	pt_byte,											\
	p_Name,												\
	p_Desc,												\
	0,													\
	0xfe,  												\
	0xff,    											\
	MPF_STATE,											\
	p_Def												\
};

#define NR_LIB_AMPLITUDE16_PAR(p_Var,p_Name,p_Desc, p_Def)	\
CMachineParameter const p_Var =								\
{															\
	pt_word,												\
	p_Name,													\
	p_Desc,													\
	0,														\
	0xfffe, 												\
	0xffff,    												\
	MPF_STATE,												\
	p_Def													\
};

#define NR_LIB_NOTE_PAR(p_Var,p_Name,p_Desc)	\
CMachineParameter const p_Var =					\
{												\
	pt_note,									\
	p_Name,										\
	p_Desc,	 									\
	NOTE_MIN,									\
	NOTE_OFF,									\
	NOTE_NO,									\
	0,											\
	NOTE_NO										\
};

#define NR_LIB_REL_FRQ_PAR(p_Var,p_Name,p_Desc,p_Def)		\
CMachineParameter const p_Var =								\
{															\
	pt_word,												\
	p_Name,													\
	p_Desc,													\
	0x0001,													\
	0xffff,													\
	0x0000,													\
	MPF_STATE,												\
	p_Def													\
};

///////////////////////////////////////////////////////////////////////////////
//
//  Base class to define standard read interface for envelopes track classes
//

class c_Envel_Track_Base
{
  protected:

    bool  f_Static;
    float f_Current;

  public:

    bool Is_Static (void)
    {
      return f_Static;
    }

    float Current_Level (void)
    {
      return f_Current;
    }

    virtual bool Overwrite (float *psamples, int numsamples) = 0;
    virtual bool Multiply  (float *psamples, int numsamples) = 0;
};

//
//  c_Decay_Global and c_Decay_Track should handle cases where the
//  attack/decay operates over a linear range of 0.0 to 1.0.
//

class c_Decay_Global
{
  private:

    friend class c_Decay_Track;

    int    f_Samples_Per_Sec;

    float  f_Attack_A;
    float  f_Attack_B;
    float  f_Decay_A;

  public:

    c_Decay_Global (int p_Samples_Per_Sec);

    void Tick (int p_Attack, int p_Decay);
};

class c_Decay_Track : public c_Envel_Track_Base
{
  private:

    const c_Decay_Global *f_Global;

    bool   f_On;      //  Note currently on?
    float  f_Set;     //  Set target - only == target when f_On == true

    float  f_Target;

  public:

    c_Decay_Track (const c_Decay_Global &p_Global);

    void Tick (int p_Note, int p_Volume, int p_Init);

	//  The following are used to allow external optimisation when amplitude is constant


    //  If the signal is static at zero, these return false WITHOUT WRITING TO THE
	//  BUFFER. If the buffer really does need clearing, use DSPLIB. In most cases,
	//  however, this indicates silence that can be ignored.

    virtual bool Overwrite (float *psamples, int numsamples);
    virtual bool Multiply  (float *psamples, int numsamples);

    void Stop (void);
};

///////////////////////////////////////////////////////////////////////////////
//
//  More standard ADSR type envelope
//
//  Linear attack  goes to     100% volume in xxxx ms
//  e^-kt  decay   goes to sustain% volume with given half life
//  e^-kt  release goes to       0%
//
//  Support soft attack as well?
//

class c_ADSR_Global
{
  private:

    friend class c_ADSR_Track;

    int    f_Samples_Per_Sec;

    int    f_Attack_Time;

    float  f_Sustain;

    float  f_Decay_A;
    float  f_Decay_B;

    float  f_Release_A;

    float  f_Attack;  //  Base values, mainly for later editing
    float  f_Decay;
    float  f_Release;

  public:

    c_ADSR_Global (void)
    {
    }

    void Set_Attack  (float p_Attack_Time  )
    {
      f_Attack      = p_Attack_Time;
      f_Attack_Time = (int) ((p_Attack_Time * (float) f_Samples_Per_Sec) / 1000.0);
    }

    void Set_Decay   (float p_Decay_Rate   )
    {
      f_Decay   = p_Decay_Rate;
      f_Decay_A = pow (2.0, -1000.0 / (p_Decay_Rate * f_Samples_Per_Sec));
      f_Decay_B = 1.0 - f_Decay_A;
    }

    void Set_Sustain (float p_Sustain_Level)
    {
      f_Sustain = p_Sustain_Level;
    }

    void Set_Release (float p_Release_Rate )
    {
      f_Release   = p_Release_Rate;
      f_Release_A = pow (2.0, -1000.0 / (p_Release_Rate * f_Samples_Per_Sec));
    }

    float Get_Attack  (void)  {  return f_Attack;   }
    float Get_Decay   (void)  {  return f_Decay;    }
    float Get_Sustain (void)  {  return f_Sustain;  }
    float Get_Release (void)  {  return f_Release;  }

    void Set_Samples_Per_Sec (int p_Samples_Per_Sec)
    {
      f_Samples_Per_Sec = p_Samples_Per_Sec;

      Set_Attack  (200.0);  //  Set some defaults
      Set_Decay   (200.0);
      Set_Sustain (  0.0);
      Set_Release (200.0);
    }
};

class c_ADSR_Track : public c_Envel_Track_Base
{
  private:

    const c_ADSR_Global *f_Global;

    int    f_Phase;   //  0 = attack, 1 = decay/sustain, 2 = release

    int    f_Attack_Count;  //  Used to time attack phase

    float  f_Previous;  //  f_Current prior to new note - used to remove clicks if previous not not finished
    float  f_Volume;

  public:

    c_ADSR_Track (const c_ADSR_Global &p_Global)
    {
      f_Global = &p_Global;

      f_Phase = 2;
      f_Static = true;
      f_Current = 0.0;
      f_Previous = 0.0;
    }

    void Set_Volume (float p_Volume)
    {
      f_Volume = p_Volume;
    }

    void Start_Note (void)
    {
      f_Phase   = 0;
      f_Static  = false;
      f_Previous = f_Current;
      f_Current = 0.0;
      f_Attack_Count = 0;
    }

    void Stop_Note (void)
    {
      f_Phase  = 2;
      f_Static = false;
    }

    void Stop (void)
    {
      f_Phase   = 2;
      f_Static  = true;
      f_Previous = 0.0;
      f_Current = 0.0;
    }

    //  If the signal is static at zero, these return false WITHOUT WRITING TO THE
    //  BUFFER. If the buffer really does need clearing, use DSPLIB. In most cases,
    //  however, this indicates silence that can be ignored.

    virtual bool Overwrite (float *psamples, int numsamples);
    virtual bool Multiply  (float *psamples, int numsamples);
};

///////////////////////////////////////////////////////////////////////////////
//
//  ADSR type envelope extended to specify start, peak and end levels - needed
//  for modulators, which do not inherently start and end at zero or peak at 1
//
//  Linear attack  goes to    peak% volume in xxxx ms
//  e^-kt  decay   goes to sustain% volume with given half life
//  e^-kt  release goes to     end%
//

class c_ADSR2_Global
{
  private:

    friend class c_ADSR2_Track;

    int    f_Samples_Per_Sec;

    int    f_Attack_Time;

    float  f_Sustain;

    float  f_Decay_A;
    float  f_Decay_B;

    float  f_Release_A;
    float  f_Release_B;

    float  f_Attack;  //  Base values, mainly for later editing
    float  f_Decay;
    float  f_Release;
    float  f_Start;
    float  f_Peak;
    float  f_End;

  public:

    c_ADSR2_Global (void)
    {
    }

    void Set_Attack  (float p_Attack_Time  )
    {
      f_Attack      = p_Attack_Time;
      f_Attack_Time = (int) ((p_Attack_Time * (float) f_Samples_Per_Sec) / 1000.0);
    }

    void Set_Decay   (float p_Decay_Rate   )
    {
      f_Decay   = p_Decay_Rate;
	    f_Decay_A = pow (2.0, -1000.0 / (p_Decay_Rate * f_Samples_Per_Sec));
      f_Decay_B = 1.0 - f_Decay_A;
    }

    void Set_Start (float p_Start_Level)
    {
      f_Start = p_Start_Level;
    }

    void Set_Peak (float p_Peak_Level)
    {
      f_Peak = p_Peak_Level;
    }

    void Set_Sustain (float p_Sustain_Level)
    {
      f_Sustain = p_Sustain_Level;
    }

    void Set_End (float p_End_Level)
    {
      f_End = p_End_Level;
    }

    void Set_Release (float p_Release_Rate )
    {
      f_Release   = p_Release_Rate;
      f_Release_A = pow (2.0, -1000.0 / (p_Release_Rate * f_Samples_Per_Sec));
      f_Release_B = 1.0 - f_Release_A;
    }

    float Get_Start   (void)  {  return f_Start;    }
    float Get_Attack  (void)  {  return f_Attack;   }
    float Get_Peak    (void)  {  return f_Peak;     }
    float Get_Decay   (void)  {  return f_Decay;    }
    float Get_Sustain (void)  {  return f_Sustain;  }
    float Get_Release (void)  {  return f_Release;  }
    float Get_End     (void)  {  return f_End;      }

    void Set_Samples_Per_Sec (int p_Samples_Per_Sec)
    {
      f_Samples_Per_Sec = p_Samples_Per_Sec;

      Set_Attack  (200.0);  //  Set some defaults
      Set_Decay   (200.0);
      Set_Sustain (  0.0);
      Set_Release (200.0);
      Set_Start   (  0.0);
      Set_Peak    (  1.0);
      Set_End     (  0.0);
    }
};

class c_ADSR2_Track : public c_Envel_Track_Base
{
  private:

    const c_ADSR2_Global *f_Global;

    int    f_Phase;   //  0 = attack, 1 = decay/sustain, 2 = release

    int    f_Attack_Count;  //  Used to time attack phase

  public:

    c_ADSR2_Track (const c_ADSR2_Global &p_Global)
    {
      f_Global  = &p_Global;

      f_Phase   = 2;
      f_Static  = true;
      f_Current = 0.0;
    }

    void Start_Note (void)
    {
      f_Phase        = 0;
      f_Static       = false;
      f_Current      = 0.0;
      f_Attack_Count = 0;
    }

    void Stop_Note (void)
    {
      f_Phase  = 2;
      f_Static = false;
    }

    void Stop (void)
    {
      f_Phase   = 2;
      f_Static  = true;
      f_Current = 0.0;
    }

    //  If the signal is static at zero, these return false WITHOUT WRITING TO THE
    //  BUFFER. If the buffer really does need clearing, use DSPLIB. In most cases,
    //  however, this indicates silence that can be ignored.

    virtual bool Overwrite (float *psamples, int numsamples);
    virtual bool Multiply  (float *psamples, int numsamples);
};

///////////////////////////////////////////////////////////////////////////////
//
//  Simplified version for fade effects on existing signals - no triggering or
//  phases. Note - use 16 bit current and target amplitude values.
//

class c_Decay_Simple : public c_Envel_Track_Base
{
private:

    int    f_Samples_Per_Sec;

    float  f_Attack_A;
    float  f_Attack_B;

    float  f_Target;

public:

    c_Decay_Simple (int p_Samples_Per_Sec);

    void Tick (int p_Attack, int p_Target, int p_Init);

    virtual bool Overwrite (float *psamples, int numsamples);
    virtual bool Multiply  (float *psamples, int numsamples);

    void Stop (void);
};

//
//  c_Pitch_Global and c_Pitch_Track should handle cases where the
//  attack/decay operates over a pitch range. The pitch is internally handled
//  using a linear octave value, and translated to a frequency for output.
//
//  Note that the pitch is output to a buffer, where it can be used sample-by-
//  sample for waveform generation. It may be realistic to have a scaled sample
//  rate for pitch handling in the future (to reduce CPU wasted on near identical
//  values), but at present the output frequency
//  is scaled by the sample rate.
//
//  Output units are cycles per sample
//

class c_Pitch_Global
{
  private:

    friend class c_Pitch_Track;

    int    f_Samples_Per_Sec;

    float  f_Attack_A;
    float  f_Attack_B;

    float  f_Attack;  //  Base value, mainly for later editing

  public:

    c_Pitch_Global (int p_Samples_Per_Sec);

    void  Set_Attack (float p_Attack)
    {
      f_Attack = p_Attack;

      f_Attack_A = pow (2.0, -1000.0 / (p_Attack * f_Samples_Per_Sec));
      f_Attack_B = 1.0 - f_Attack_A;
    }

    float Get_Attack (void)
    {
      return f_Attack;
    }

    void Tick (int p_Attack);
};

class c_Pitch_Track : public c_Envel_Track_Base
{
  private:

    const c_Pitch_Global *f_Global;

    float  f_Target;

    float  f_Frq_Scale;

  public:

    c_Pitch_Track (const c_Pitch_Global &p_Global);

    void Tick (int p_Note, int p_Init);

    virtual bool Overwrite (float *psamples, int numsamples);
    virtual bool Multiply  (float *psamples, int numsamples);

    void Stop (void);
};

///////////////////////////////////////////////////////////////////////////////

void NR_DSP_Multiply (float *p_Dest, float *p_Scale, int n);
void NR_DSP_Multiply (float *p_Dest, float *p_Scale, float p_Const_Scale, int n);

void NR_DSP_Multiply (float *p_Dest, float *p_Src, float *p_Scale, int n);

///////////////////////////////////////////////////////////////////////////////
//  Some functions are so standard, there's no point having them in every
//  machines source code.

#define NR_STD_GENERATOR_WORK											\
bool mi::Work(float *psamples, int numsamples, int const mode)			\
{																		\
	if ((mode & WM_WRITE) == 0)											\
	{																	\
		return false;													\
	}																	\
																		\
	bool gotsomething = false;											\
	bool l_Temp;														\
																		\
	float *paux = pCB->GetAuxBuffer();									\
																		\
	for (int c = 0; c < numTracks; c++)									\
	{																	\
		if (gotsomething)												\
		{																\
			l_Temp = Tracks[c]->Generate(paux, numsamples);				\
																		\
		    gotsomething |= l_Temp;										\
																		\
			if (l_Temp)													\
			{															\
				DSP_Add(psamples, paux, numsamples);					\
			}															\
		}																\
		else															\
		{																\
			gotsomething |= Tracks[c]->Generate(psamples, numsamples);	\
		}																\
	}																	\
																		\
	return gotsomething;												\
}

///////////////////////////////////////////////////////////////////////////////

#define NR_STD_GENERATOR_SET_NUM_TRACKS									\
void mi::SetNumTracks(int const n)										\
{																		\
	for (int i = numTracks; i < n; i++)									\
	{																	\
		Tracks[i]->Reset ();											\
	}																	\
																		\
	numTracks = n;														\
}

///////////////////////////////////////////////////////////////////////////////

#define NR_STD_GENERATOR_STOP											\
void mi::Stop()															\
{																		\
	for (int c = 0; c < numTracks; c++)									\
	{																	\
		Tracks[c]->Stop();												\
	}																	\
}

///////////////////////////////////////////////////////////////////////////////

#endif
